using System;
using System.Data;
using System.Collections;

using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stas Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/27/2005</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Encapsulates client functionality for access to VistA laboratory test data.</summary>

	#endregion

	public class LaboratoryTest
	{
		private const string LABORATORY_TEST_LOOKUP_RPC_NAME = "VBECS LABORATORY TEST LOOKUP";
		private const string LABORATORY_TEST_RESULT_LOOKUP_RPC_NAME = "VBECS LAB TEST RESULTS LOOKUP";
		private const string LABORARORY_TEST_TABLENAME = "LabTest";

		private const string VISTA_DATE_FORMAT = "yyyyMMdd";

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4534"> 
		///		<ExpectedInput>'HEM' as the testName parameter.</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with Lab tests return from VistA.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4535"> 
		///		<ExpectedInput>'null' as the testName parameter.</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Static method to search for Laboratory Tests from the Vista file #60 based on input string testName parameter.
		/// </summary>
		/// <param name="testName">String representing a Laboratory Test name to perform the search</param>
		/// <returns><see cref="DataTable"/> containing Lab Tests that match the input criteria</returns>
		public static DataTable GetLaboratoryTestData( string testName )
		{
			if( testName == null )
				throw( new ArgumentNullException( "testName" ) );

			DataSet _ds = new DataSet();

			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetLaboratoryTestRpcRequest( testName ) );

			_ds = DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );

			return _ds.Tables.Contains( LABORARORY_TEST_TABLENAME ) ? _ds.Tables[LABORARORY_TEST_TABLENAME] : _ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Brian Tomlin</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/24/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5457"> 
		///		<ExpectedInput>startDate = 01/01/2000, endDate = DateTime.Now, divisionCode = null, tests = 3 and 4, patients = Array of VistAPatientId's</ExpectedInput>
		///		<ExpectedOutput>Non-empty DataTable with All Lab test results return from VistA.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5458"> 
		///		<ExpectedInput>patients = ArrayList with no values</ExpectedInput>
		///		<ExpectedOutput>ArgumentNullException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Static method used to obtain a list of Laboratory test results in VistA for a patient or multiple patients.
		/// </summary>
		/// <param name="startDate"><see cref="DateTime"/> of starting date range to perform search</param>
		/// <param name="endDate"><see cref="DateTime"/> of ending date range to perform search</param>
		/// <param name="divisionCode">Division code string</param>
		/// <param name="tests"><see cref="ArrayList"/> of VistA Laboratory Test file #60 internal entry numbers</param>
		/// <param name="patients"><see cref="ArrayList"/> of VistA Patient Id's</param>
		/// <returns><see cref="DataTable"/> containing Laboratory Test results for selected patient(s) from the VistA Lab system</returns>
		public static DataSet GetLaboratoryTestResults( DateTime startDate, DateTime endDate, string divisionCode, ArrayList tests, ArrayList patients)
		{
			if( patients.Count < 1 )
				throw( new ArgumentNullException( "patients" ) );

			string sDate = ( !(startDate > DateTime.Now) || !(startDate == DateTime.MinValue) ) ? startDate.ToString( VISTA_DATE_FORMAT ) : null;
			string eDate = ( !(endDate < startDate) || !(endDate > DateTime.Now) ) ? DateTime.Now.ToString( VISTA_DATE_FORMAT ) : null;

			string _rpcResult = VistALink.GetRpcBroker().ExecuteRpc( GetLabTestResultRpcRequest( sDate, eDate, divisionCode, tests, patients ) );

			DataSet ds = DataTransformUtility.LoadXmlStringIntoDataSet( _rpcResult );
			foreach( DataTable dt in ds.Tables )
			{
				if( dt.TableName == LABORARORY_TEST_TABLENAME )
				{
					dt.Columns.Add( new DataColumn( "ResultDateTime", typeof(DateTime) ) );
					foreach( DataRow dr in dt.Rows )
					{
						if( dr.Table.Columns.Contains("TestDate") && !dr.IsNull("TestDate") )
						{
							dr["ResultDateTime"] = Common.HL7DateFormat.ConvertHL7DateTime( dr["TestDate"].ToString() );
						}
					}
				}
			}

			return ds;
		}

		/// <summary>
		/// Creates the lab test lookup RPC request from given searchValue parameter.
		/// </summary>
		/// <param name="searchValue">String used to perform a lookup of matching Lab Tests.</param>
		/// <returns>Laboratory Test RpcRequest object</returns>
		private static RpcRequest GetLaboratoryTestRpcRequest(string searchValue)
		{
			RpcRequest _labTestRequest = VistALink.CreateRpcRequest( LABORATORY_TEST_LOOKUP_RPC_NAME );
			_labTestRequest.Parameters.Add( new RpcStringParameter( 1, ( searchValue != null ) ? searchValue : String.Empty ) );

			return _labTestRequest;
		}

		/// <summary>
		/// Creates the lab test result lookup RPC request from given parameters.
		/// </summary>
		/// <param name="sDate">String date to begin search.</param>
		/// <param name="eDate">String date to end search.</param>
		/// <param name="div">Division Code</param>
		/// <param name="tests"><see cref="ArrayList"/> of Laboratory Test internal entry numbers to search for results.</param>
		/// <param name="patients"><see cref="ArrayList"/> of VistA Patient Id's to search for result.</param>
		/// <returns>Laboratory Test Results RpcRequest object</returns>
		private static RpcRequest GetLabTestResultRpcRequest( string sDate, string eDate, string div, ArrayList tests, ArrayList patients )
		{
			RpcRequest _labTestResultRequest = VistALink.CreateRpcRequest( LABORATORY_TEST_RESULT_LOOKUP_RPC_NAME );
			_labTestResultRequest.Parameters.Add( new RpcStringParameter( 1, ( sDate != null ) ? sDate : String.Empty ) );
			_labTestResultRequest.Parameters.Add( new RpcStringParameter( 2, ( eDate != null ) ? eDate : String.Empty ) );
			_labTestResultRequest.Parameters.Add( new RpcStringParameter( 3, ( div != null ) ? div : String.Empty ) );
			_labTestResultRequest.Parameters.Add( GetRpcArrayParameter( 4, tests ) );
			_labTestResultRequest.Parameters.Add( GetRpcArrayParameter( 5, patients ) );

			return _labTestResultRequest;
		}

		private static RpcArrayParameter GetRpcArrayParameter( int index, ArrayList arr )
		{
			Hashtable _htBuf = new Hashtable();
			for( int i=0; i < arr.Count; i++ )
			{
				_htBuf.Add( Convert.ToString(i+1), arr[i].ToString() );
			}
			return new RpcArrayParameter( index, _htBuf );
		}
	}
}
